<?php

namespace Fatpanda\BambooConnector\HttpFoundation;

class Request
{
    /** @var string  */
    protected $url;

    /** @var string  */
    private $requestMethod;

    /** @var array */
    private $query;

    /** @var string  */
    private $locale = null;

    const HTTP_GET = 'GET';
    const HTTP_POST = 'POST';
    const HTTP_PATCH = 'PATCH';
    const HTTP_PUT = 'PUT';
    const HTTP_DELETE = 'DELETE';

    const HTTP_METHODS = [
        self::HTTP_GET,
        self::HTTP_POST,
        self::HTTP_PATCH,
        self::HTTP_PUT,
        self::HTTP_DELETE,
    ];

    /**
     * Request constructor.
     * @param string $url
     * @param string $requestMethod
     * @param array $query
     */
    public function __construct(string $url, string $requestMethod = self::HTTP_GET, array $query = [])
    {
        $this->url = $url;

        if (!in_array($requestMethod, self::HTTP_METHODS)) {
            throw new \InvalidArgumentException(sprintf('%s is not a supported request method', $requestMethod));
        }

        $this->requestMethod = $requestMethod;
        $this->query = $query;
    }

    /**
     * @return string
     */
    public function getUrl()
    {
        return $this->url;
    }

    /**
     * @return string
     */
    public function getRequestMethod()
    {
        return $this->requestMethod;
    }

    /**
     * @return null|string
     */
    public function getLocale()
    {
        return $this->locale;
    }

    /**
     * @param string $locale
     * @return $this
     */
    public function setLocale(string $locale)
    {
        $this->locale = $locale;

        return $this;
    }

    /**
     * @return array
     */
    public function getQuery(): array
    {
        return $this->query;
    }

    /**
     * @param array $query
     */
    public function setQuery(array $query)
    {
        $this->query = $query;
    }

    /**
     * @return bool
     */
    public function isCacheable()
    {
        return method_exists($this, 'hasCacheEnabled') && empty($this->query);
    }

    /**
     * @return bool
     */
    public function isPayloadable()
    {
        return method_exists($this, 'setPayload');
    }

    /**
     * @return bool
     */
    public function isPaginatable()
    {
        return method_exists($this, 'getRangeHeaders');
    }
}