<?php

namespace Tests\Fatpanda\BambooConnector;

use Fatpanda\BambooConnector\HttpFoundation\DeleteRequest;
use Fatpanda\BambooConnector\HttpFoundation\GetRequest;
use Fatpanda\BambooConnector\HttpFoundation\PatchRequest;
use Fatpanda\BambooConnector\HttpFoundation\PostRequest;
use Fatpanda\BambooConnector\HttpFoundation\PutRequest;
use Fatpanda\BambooConnector\HttpFoundation\Request;
use PHPUnit\Framework\TestCase;

class RequestTest extends TestCase
{
    /** @dataProvider providerRequestTypes */
    public function test($class, $config)
    {
        if (!is_null($config['payload'])) {
            $request = new $class('testurl', $config['payload']);
        } else {
            $request = new $class('testurl');
        }

        $this->assertInstanceOf(Request::class, $request);

        $this->assertEquals('testurl', $request->getUrl());
        $this->assertNull($request->getLocale());
        $this->assertEquals($config['request_method'], $request->getRequestMethod());

        // Traits
        $config['cacheable'] ? $this->assertTrue($request->isCacheable()) : $this->assertFalse($request->isCacheable());
        $config['paginatable'] ? $this->assertTrue($request->isPaginatable()) : $this->assertFalse($request->isPaginatable());
        $config['payloadable'] ? $this->assertTrue($request->isPayloadable()) : $this->assertFalse($request->isPayloadable());

        // Payload
        if ($request->isPayloadable()) {
            $this->assertArrayHasKey('sub', $request->getPayload());
            $this->assertEquals('value', $request->getPayload()['sub']['test']);
            $this->assertArrayHasKey('top', $request->getPayload());
            $this->assertEquals('test', $request->getPayload()['top']);
        }

        // Cache
        if ($request->isCacheable()) {
            $this->assertNull($request->hasCacheEnabled());

            $request->enableCache();
            $this->assertTrue($request->hasCacheEnabled());

            $request->disableCache();
            $this->assertFalse($request->hasCacheEnabled());
        }

        // Pagination
        if ($request->isPaginatable()) {
            $this->assertNull($request->getPage());

            $request->setPage(5);
            $this->assertEquals(5, $request->getPage());
            $this->assertEquals('100-124', $request->getRangeString());
            $this->assertEquals([
                'Range' => '100-124',
                'Range-Unit' => 'items'
            ], $request->getRangeHeaders());

            $request->setPage(3);
            $this->assertEquals(3, $request->getPage());
            $this->assertEquals('50-74', $request->getRangeString());
            $this->assertEquals([
                'Range' => '50-74',
                'Range-Unit' => 'items'
            ], $request->getRangeHeaders());

            $request->setPage(2);
            $request->setCollectionCount(9);
            $this->assertEquals(9, $request->getCollectionCount());
            $this->assertEquals('9-17', $request->getRangeString());
            $this->assertEquals([
                'Range' => '9-17',
                'Range-Unit' => 'items'
            ], $request->getRangeHeaders());

            // Page should automatically be assigned '1' when setting collection count
            $request = new $class('testurl');
            $request->setCollectionCount(4);
            $this->assertEquals(1, $request->getPage());

            // Collection count should automatically be assigned '25' when setting page
            $request = new $class('testurl');
            $request->setPage(1);
            $this->assertEquals(25, $request->getCollectionCount());
        }

        // Locale
        $request->setLocale('en');
        $this->assertEquals('en', $request->getLocale());
    }

    public function test_directInstantiation()
    {
        $request = new Request('test', 'GET');
        $this->assertEquals('GET', $request->getRequestMethod());
        $this->assertNull($request->getLocale());
    }

    public function test_directInstantiationInvalidHttpMethod()
    {
        $this->expectException(\InvalidArgumentException::class);
        $request = new Request('test', 'INVALID');
    }

    public function providerRequestTypes()
    {
        return [
            'get' => [
                GetRequest::class,
                [
                    'payload' => null,
                    'request_method' => Request::HTTP_GET,
                    'cacheable' => true,
                    'paginatable' => true,
                    'payloadable' => false
                ]
            ],
            'delete' => [
                DeleteRequest::class,
                [
                    'payload' => null,
                    'request_method' => Request::HTTP_DELETE,
                    'cacheable' => false,
                    'paginatable' => false,
                    'payloadable' => false
                ]
            ],
            'put' => [
                PutRequest::class,
                [
                    'payload' => [
                        'sub' => [
                            'test' => 'value'
                        ],
                        'top' => 'test'
                    ],
                    'request_method' => Request::HTTP_PUT,
                    'cacheable' => false,
                    'paginatable' => false,
                    'payloadable' => true
                ]
            ],
            'post' => [
                PostRequest::class,
                [
                    'payload' => [
                        'sub' => [
                            'test' => 'value'
                        ],
                        'top' => 'test'
                    ],
                    'request_method' => Request::HTTP_POST,
                    'cacheable' => false,
                    'paginatable' => false,
                    'payloadable' => true
                ]
            ],
            'patch' => [
                PatchRequest::class,
                [
                    'payload' => [
                        'sub' => [
                            'test' => 'value'
                        ],
                        'top' => 'test'
                    ],
                    'request_method' => Request::HTTP_PATCH,
                    'cacheable' => false,
                    'paginatable' => false,
                    'payloadable' => true
                ]
            ]
        ];
    }
}