<?php

namespace Tests\Fatpanda\BambooConnector;

use Fatpanda\BambooConnector\HttpFoundation\GetRequest;
use Fatpanda\BambooConnector\HttpFoundation\PostRequest;
use Fatpanda\BambooConnector\HttpFoundation\PutRequest;
use Fatpanda\BambooConnector\HttpFoundation\Request;
use GuzzleHttp\Client as GuzzleClient;
use GuzzleHttp\Psr7\Response;
use Fatpanda\BambooConnector\HttpFoundation\Response as BambooResponse;
use PHPUnit\Framework\TestCase;
use Fatpanda\BambooConnector\BambooConnector;
use Psr\SimpleCache\CacheInterface;

class BambooConnectorTest extends TestCase
{
    /** @var GuzzleClient|\PHPUnit_Framework_MockObject_MockObject */
    private $mockGuzzleClient;

    /** @var CacheInterface|\PHPUnit_Framework_MockObject_MockObject  */
    private $mockCacheHandler;

    private $apiKey = 'test-api-key';

    // ---------------------
    // setup/helper methods:
    // ---------------------

    public function setUp()
    {
        // set up the mock objects, those will be needed in all tests,
        // as they are required for BambooConnector instantiation
        $this->mockGuzzleClient = $this->getMockBuilder(GuzzleClient::class)->getMock();
        $this->mockCacheHandler = $this->getMockBuilder(CacheInterface::class)->getMock();
    }

    /**
     * If you have request/cache expectations to set up, be sure to do that before you build the connector!
     * @param bool $registerCacheHandler
     * @return BambooConnectorMock
     */
    private function buildMockConnector($registerCacheHandler = false)
    {
        $connector = new BambooConnectorMock(
            $this->mockGuzzleClient,
            $this->apiKey
        );

        if ($registerCacheHandler) {
            $connector->setCacheHandler($this->mockCacheHandler);
        }

        return $connector;
    }

    /**
     * Setup the expectation that the GuzzleHttp\Client will be called a specific number of times (!) with the given values
     * (and return specific data, if given)
     *
     * @param int $count
     * @param string $method
     * @param string $uri
     * @param array $responseData
     * @param array $guzzleOptions
     */
    private function expectResponse($count, $method, $uri, $responseData = ['some' => 'data'], $guzzleOptions = [])
    {
        $response = $this->getMockBuilder(Response::class)
            ->disableOriginalConstructor()
            ->getMock();

        $response->expects($this->exactly($count))->method('getBody')->will($this->returnValue(
            \GuzzleHttp\json_encode($responseData)
        ));

        // the auth header should always be there in the options when the guzzle client is called
        if (!array_key_exists('headers', $guzzleOptions)) {
            $guzzleOptions['headers'] = [];
        }
        $guzzleOptions['headers']['X-Authorization'] = 'Bearer ' . $this->apiKey;

        $this->mockGuzzleClient->expects($this->exactly($count))->method('request')->with(
            $this->equalTo($method),
            $this->equalTo($uri),
            $this->equalTo($guzzleOptions)
        )->will($this->returnValue($response));
    }

    // -----------------
    // tests start here:
    // -----------------

    public function test_getters()
    {
        $connector = $this->buildMockConnector(false);

        $this->assertEquals('de', $connector->getLocale());
        $connector->setLocale('en');
        $this->assertEquals('en', $connector->getLocale());

        $this->assertNull($connector->getCacheHandler());
        $this->assertFalse($connector->hasCacheEnabled());
        $connector->setCacheHandler($this->mockCacheHandler);
        $this->assertInstanceOf(CacheInterface::class, $connector->getCacheHandler());
        $this->assertTrue($connector->hasCacheEnabled());

        $connector = $this->buildMockConnector(true);
        $this->assertTrue($connector->hasCacheEnabled());
        $this->assertInstanceOf(CacheInterface::class, $connector->getCacheHandler());
        $this->assertTrue($connector->hasCacheEnabled());

        $connector = $this->buildMockConnector(false);
        $this->assertFalse($connector->hasCacheEnabled());
        $connector->enableCache($this->mockCacheHandler);
        $this->assertInstanceOf(CacheInterface::class, $connector->getCacheHandler());
        $this->assertTrue($connector->hasCacheEnabled());

        $connector = $this->buildMockConnector(false);
        $this->assertFalse($connector->hasCacheEnabled());
        $this->expectException(\LogicException::class);
        $connector->enableCache();
    }

    // should build the cache key from request method and uri
    public function test_getCacheKey()
    {
        $request = new GetRequest('aURL');

        $connector = $this->buildMockConnector();
        $this->assertEquals(
            'bamboo_c691088e75227115257c553c266143bef3a1d63f2899217ded266ae9143d494f',
            $connector->getCacheKey($request, 'de/api/aUrl')
        );

        $request->setLocale('en');
        $this->assertEquals(
            'bamboo_8de93acb8906164fff46af7cd297ad3700e9a1b4273d655b46ad68aa236ec5d7',
            $connector->getCacheKey($request, 'en/api/aUrl')
        );

        $request->setPage(5);
        $request->setCollectionCount(2);
        $this->assertEquals(
            'bamboo_7e7e1a499bd26ef952eb4bc7e2e977d53c36eed676957012924dd51a5892c506',
            $connector->getCacheKey($request, 'en/api/aUrl')
        );
    }

    // adding auth header to guzzle options should leave the rest of the options array intact
    public function test_addHeaders()
    {
        $connector = $this->buildMockConnector();
        $request = new GetRequest('aURL');

        // on empty stomach
        $options = [];
        $connector->addHeaders($options, $request);
        $this->assertEquals(['headers' => ['X-Authorization' => 'Bearer ' . $this->apiKey]], $options);

        // other auth header already exists
        $options = ['headers' => ['X-Authorization' => 'Bearer 123']];
        $connector->addHeaders($options, $request);
        $this->assertEquals([
            'headers' => [
                'X-Authorization' => 'Bearer '.$this->apiKey
            ]
        ], $options);

        // other options already exist
        $options = ['json' => ['dataalreadyset' => 'data']];
        $request = new PostRequest('aURL', ['some' => 'data']);
        $connector->addHeaders($options, $request);
        $this->assertEquals([
            'json' => ['some' => 'data'],
            'headers' => ['X-Authorization' => 'Bearer '.$this->apiKey]
        ], $options);

        // with pagination defaults
        $options = ['json' => ['some' => 'data']];
        $request = new GetRequest('aURL');
        $request->setPage(3);
        $request->setCollectionCount(4);
        $connector->addHeaders($options, $request);
        $this->assertEquals([
            'json' => ['some' => 'data'],
            'headers' => [
                'X-Authorization' => 'Bearer ' . $this->apiKey,
                'Range' => '8-11',
                'Range-Unit' => 'items',
            ]
        ], $options);
    }

    // a basic request without caching
    public function test_request_success()
    {
        $data = ['some' => 'data'];
        $request = new GetRequest('test/url');

        // calling twice should do the actual request twice
        $this->expectResponse(2, 'GET', 'de/api/test/url', $data);

        $connector = $this->buildMockConnector(false);
        $this->assertInstanceOf(BambooResponse::class, $connector->sendRequest($request));
        $this->assertInstanceOf(BambooResponse::class, $connector->sendRequest($request));
    }

    // a basic request with caching enabled
    public function test_request_caching()
    {
        $data = ['some' => 'data'];
        $request = new GetRequest('testurl');

        $serializedResponse = 'C:48:"Fatpanda\BambooConnector\HttpFoundation\Response":51:{a:4:{i:0;N;i:1;N;i:2;s:15:"{"some":"data"}";i:3;N;}}';
        $cacheKey = 'bamboo_6ee525720cc9478ad5ab1501b52c042e3efdd9664dceca26b7ac9e7eb9fbbe8c';

        // calling twice should only do the request once, and then use the cached response the second time
        $this->expectResponse(1, 'GET', 'de/api/testurl', $data);

        // the cache entry will be set once (on the first call)
        $this->mockCacheHandler->expects($this->once())->method('set')->with($cacheKey, $serializedResponse)->will(
            $this->returnValue(true)
        );

        // the cache entry will be read only on the second request
        $this->mockCacheHandler->expects($this->exactly(1))->method('get')->with($cacheKey)->will(
            $this->returnValue($serializedResponse)
        );

        // The cache will be checked twice if it has the cacheKey set
        $this->mockCacheHandler->expects($this->exactly(2))->method('has')->with($cacheKey)->will(
            $this->onConsecutiveCalls(null, true));

        $connector = $this->buildMockConnector(true);
        $this->assertInstanceOf(BambooResponse::class, $connector->sendRequest($request));
        $this->assertInstanceOf(BambooResponse::class, $connector->sendRequest($request));
    }

    // a request with a payload
    public function test_request_payload()
    {
        $data = ['some' => 'data'];
        $request = new PutRequest('testurl', $data);

        // the payload will be passed on to guzzle under the 'json' option
        $this->expectResponse(
            1, 'PUT', 'de/api/testurl', [],
            ['json' => $data]
        );
        $connector = $this->buildMockConnector();
        $this->assertInstanceOf(BambooResponse::class, $connector->sendRequest($request));
    }

    // a request with pagination
    public function test_request_pagination()
    {
        // setup all the expected data ...
        $cacheKey1 = 'bamboo_9395a9d7b3e4ad755f1d218f68d9a56b78bc6eb964e020b040a05880df8a77b0';
        $cacheKey2 = 'bamboo_5de11320f528b128e37def9c8ea46fe83653da084a71de3d270ae700480129f1';
        $data1 = 'C:48:"Fatpanda\BambooConnector\HttpFoundation\Response":52:{a:4:{i:0;N;i:1;N;i:2;s:16:"{"test":"10-19"}";i:3;N;}}';
        $data2 = 'C:48:"Fatpanda\BambooConnector\HttpFoundation\Response":52:{a:4:{i:0;N;i:1;N;i:2;s:16:"{"test":"20-29"}";i:3;N;}}';

        $body1 = \GuzzleHttp\json_encode(['test' => '10-19']);
        $body2 = \GuzzleHttp\json_encode(['test' => '20-29']);

        $guzzleOptions1 = [
            'headers' => [
                'X-Authorization' => 'Bearer ' . $this->apiKey,
                'Range' => '10-19',
                'Range-Unit' => 'items'
            ]
        ];
        $guzzleOptions2 = [
            'headers' => [
                'X-Authorization' => 'Bearer ' . $this->apiKey,
                'Range' => '20-29',
                'Range-Unit' => 'items'
            ]
        ];

        // set up expectations per hand, as they're a bit more complex
        $response = $this->getMockBuilder(Response::class)
            ->disableOriginalConstructor()
            ->getMock();

        // two requests should happen, with two differend range headers
        $response->expects($this->exactly(2))->method('getBody')
            ->will($this->onConsecutiveCalls($body1, $body2));
        $this->mockGuzzleClient->expects($this->exactly(2))->method('request')->withConsecutive(
            [$this->equalTo('GET'),$this->equalTo('de/api/testurl'),$this->equalTo($guzzleOptions1)],
            [$this->equalTo('GET'),$this->equalTo('de/api/testurl'),$this->equalTo($guzzleOptions2)]
        )->will($this->returnValue($response));

        // both responses should be saved under distinct cache keys
        $this->mockCacheHandler->expects($this->exactly(2))->method('set')
            ->withConsecutive(
                [$cacheKey1, $data1],
                [$cacheKey2, $data2]
            );

        // activating pagination and setting page length 10 should get used both times now
        $connector = $this->buildMockConnector(true);
        $request = new GetRequest('testurl');
        $request->setPage(2);
        $request->setCollectionCount(10);
        $connector->sendRequest($request);

        $request->setPage(3);
        $connector->sendRequest($request);
    }

    // missing cache handler
    public function test_request_errorNoCacheHandler()
    {
        // guzzle request should never get sent
        $this->mockGuzzleClient->expects($this->never())->method('request');
        $connector = new BambooConnectorMock(
            $this->mockGuzzleClient,
            $this->apiKey
        );

        $request = new GetRequest('test');
        $request->enableCache();

        $this->expectException(\LogicException::class);
        $connector->sendRequest($request);
    }

    // a request using the GET shorthand
    public function test_get()
    {
        $data = ['some' => 'data'];
        $this->expectResponse(1, Request::HTTP_GET, 'en/api/test/test', $data);

        $connector = $this->buildMockConnector();
        $connector->setLocale('en');
        $this->assertEquals($data, $connector->get('test/test')->getData());
    }

    // a request with the pagination shorthand
    public function test_request_getPage()
    {
        // should still add the right headers
        $data = ['test' => '25-49'];
        $this->expectResponse(1, 'GET', 'de/api/test/pagination', $data, [
            'headers' => [
                'Range' => '25-49',
                'Range-Unit' => 'items'
            ]
        ]);

        $connector = $this->buildMockConnector();
        $this->assertEquals(
            $data,
            $connector->getPage('test/pagination', 2)->getData()
        );
    }

    // a request using the DELETE shorthand
    public function test_delete()
    {
        $data = ['some' => 'data'];
        $this->expectResponse(1, 'DELETE', 'en/api/test/test', $data);

        $connector = $this->buildMockConnector();
        $connector->setLocale('en');
        $this->assertEquals($data, $connector->delete('test/test')->getData());
    }

    // a request using the PUT shorthand
    public function test_put()
    {
        $data = ['some' => 'data'];
        $this->expectResponse(1, 'PUT', 'en/api/test/test', $data, [
            'json' => $data
        ]);

        $connector = $this->buildMockConnector();
        $connector->setLocale('en');
        $this->assertEquals($data, $connector->put('test/test', $data)->getData());
    }

    // a request using the PATCH shorthand
    public function test_patch()
    {
        $data = ['some' => 'data'];
        $this->expectResponse(1, 'PATCH', 'en/api/test/test', $data, [
            'json' => $data
        ]);

        $connector = $this->buildMockConnector();
        $connector->setLocale('en');
        $this->assertEquals($data, $connector->patch('test/test', $data)->getData());
    }

    // a request using the POST shorthand
    public function test_post()
    {
        $data = ['some' => 'data'];
        $this->expectResponse(1, 'POST', 'en/api/test/test', $data, [
            'json' => $data
        ]);

        $connector = $this->buildMockConnector();
        $connector->setLocale('en');
        $this->assertEquals($data, $connector->post('test/test', $data)->getData());
    }
}

// Override connector class to make protected function logic testable
class BambooConnectorMock extends BambooConnector
{
    public function __construct(GuzzleClient $client, string $apiToken)
    {
        parent::__construct($client, $apiToken);
    }

    public function setCacheHandler(CacheInterface $cacheHandler)
    {
        return parent::setCacheHandler($cacheHandler); // TODO: Change the autogenerated stub
    }

    public function getLocale()
    {
        return parent::getLocale(); // TODO: Change the autogenerated stub
    }

    public function setLocale(string $locale)
    {
        return parent::setLocale($locale); // TODO: Change the autogenerated stub
    }

    public function enableCache(CacheInterface $cacheHandler = null)
    {
        return parent::enableCache($cacheHandler); // TODO: Change the autogenerated stub
    }

    public function disableCache()
    {
        return parent::disableCache(); // TODO: Change the autogenerated stub
    }

    public function getCacheHandler()
    {
        return parent::getCacheHandler(); // TODO: Change the autogenerated stub
    }

    public function sendRequest(Request $request)
    {
        return parent::sendRequest($request); // TODO: Change the autogenerated stub
    }

    public function get(string $url)
    {
        return parent::get($url); // TODO: Change the autogenerated stub
    }

    public function getPage(string $url, int $page = 1)
    {
        return parent::getPage($url, $page); // TODO: Change the autogenerated stub
    }

    public function post(string $url, array $payload = null)
    {
        return parent::post($url, $payload); // TODO: Change the autogenerated stub
    }

    public function put(string $url, array $payload = null)
    {
        return parent::put($url, $payload); // TODO: Change the autogenerated stub
    }

    public function patch(string $url, array $payload = null)
    {
        return parent::patch($url, $payload); // TODO: Change the autogenerated stub
    }

    public function delete(string $url)
    {
        return parent::delete($url); // TODO: Change the autogenerated stub
    }

    public function hasCacheEnabled()
    {
        return parent::hasCacheEnabled(); // TODO: Change the autogenerated stub
    }

    public function addHeaders(array &$guzzleOptions, Request $request)
    {
        parent::addHeaders($guzzleOptions, $request); // TODO: Change the autogenerated stub
    }

    public function getCacheKey(Request $request, $endpointUrl)
    {
        return parent::getCacheKey($request, $endpointUrl); // TODO: Change the autogenerated stub
    }

}