<?php

namespace Fatpanda\BambooConnector;

use Fatpanda\BambooConnector\HttpFoundation\Request,
    Fatpanda\BambooConnector\HttpFoundation\Response;
use Fatpanda\BambooConnector\HttpFoundation\GetRequest,
    Fatpanda\BambooConnector\HttpFoundation\PostRequest,
    Fatpanda\BambooConnector\HttpFoundation\PutRequest,
    Fatpanda\BambooConnector\HttpFoundation\PatchRequest,
    Fatpanda\BambooConnector\HttpFoundation\DeleteRequest;
use GuzzleHttp\Client as GuzzleClient;
use GuzzleHttp\Exception\TransferException;
use Psr\SimpleCache\CacheInterface;

class BambooConnector
{
    /** @var GuzzleClient*/
    private $client;

    /** @var string */
    private $apiToken;

    /** @var CacheInterface */
    private $cacheHandler;

    /** @var string */
    private $locale = 'de';

    /** @var bool */
    private $cacheEnabled = false;

    /**
     * @param GuzzleClient $client
     * @param string $apiToken
     */
    public function __construct(GuzzleClient $client, string $apiToken)
    {
        $this->client = $client;
        $this->apiToken = $apiToken;
    }

    /**
     * @param CacheInterface $cacheHandler
     * @return BambooConnector
     */
    public function setCacheHandler(CacheInterface $cacheHandler)
    {
        return $this->enableCache($cacheHandler);
    }

    /**
     * @return string
     */
    public function getLocale()
    {
        return $this->locale;
    }

    /**
     * @param string $locale
     * @return $this
     */
    public function setLocale(string $locale)
    {
        $this->locale = $locale;

        return $this;
    }

    /**
     * @param CacheInterface|null $cacheHandler
     * @return $this
     */
    public function enableCache(CacheInterface $cacheHandler = null)
    {
        if ($cacheHandler) {
            $this->cacheHandler = $cacheHandler;
        }

        if (!$this->cacheHandler) {
            throw new \LogicException('Cache cannot be enabled without a cache handler');
        }

        $this->cacheEnabled = true;

        return $this;
    }

    /**
     * @return $this
     */
    public function disableCache()
    {
        $this->cacheEnabled = false;

        return $this;
    }

    /**
     * @return bool
     */
    public function hasCacheEnabled()
    {
        return $this->cacheEnabled;
    }

    /**
     * @return CacheInterface
     */
    public function getCacheHandler()
    {
        return $this->cacheHandler;
    }

    /**
     * @param Request $request
     * @return Response
     * @throws \Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function sendRequest(Request $request)
    {
        $useCache = false;
        if ($request->isCacheable()) {
            $useCache = true === $request->hasCacheEnabled() || true === $this->cacheEnabled ? true : false;
        }

        if ($useCache && !$this->cacheHandler) {
            throw new \LogicException('No cache handler set for a request that relies on using it');
        }

        $locale = null !== $request->getLocale() ? $request->getLocale() : $this->locale;
        $endpointUrl = $locale . '/api/' . $request->getUrl();

        if ($useCache) {
            $cacheKey = $this->getCacheKey($request, $endpointUrl);

            if ($this->cacheHandler->has($cacheKey)) {
                $response = unserialize($this->cacheHandler->get($cacheKey));

                return $response;
            }
        }

        // add headers to guzzle header options
        $guzzleOptions = [];
        $this->addHeaders($guzzleOptions, $request);

        try {
            $guzzleResponse = $this->client->request(
                $request->getRequestMethod(),
                $endpointUrl,
                $guzzleOptions
            );
        } catch (TransferException $e) {
            $response = new \GuzzleHttp\Psr7\Response(500, ['Content-Type' => 'application/problem+json'], \GuzzleHttp\json_encode([
                [
                    'status' => 503,
                    'type' => 'client_error',
                    'title' => 'Service Unavailable',
                    'detail' => $e->getMessage()
                ]
            ]));

            return new Response($response);
        }

        $response = new Response($guzzleResponse);
        unset($guzzleResponse);

        if (isset($cacheKey)) {
            $this->cacheHandler->set($cacheKey, serialize($response), 60);
        }

        return $response;
    }

    /**
     * Shorthand for GET-requests
     * @param string $url
     * @return Response
     * @throws \Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function get(string $url)
    {
        $request = new GetRequest($url);

        return $this->sendRequest($request);
    }

    /**
     * Shorthand for paginated GET-requests ($page is 1-indexed)
     * @param string $url
     * @param int $page
     * @return Response
     * @throws \Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function getPage(string $url, int $page = 1)
    {
        if ($page < 1) {
            throw new \InvalidArgumentException('Page must be a positive integer.');
        }

        $request = new GetRequest($url);
        $request->setPage($page);

        return $this->sendRequest($request);
    }

    /**
     * Shorthand for POST-requests
     * @param string $url
     * @param array $payload
     * @return Response
     * @throws \Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function post(string $url, array $payload = null)
    {
        $request = new PostRequest($url, $payload);

        return $this->sendRequest($request);
    }

    /**
     * Shorthand for PUT-requests
     * @param string $url
     * @param array $payload
     * @return Response
     * @throws \Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function put(string $url, array $payload = null)
    {
        $request = new PutRequest($url, $payload);

        return $this->sendRequest($request);
    }

    /**
     * Shorthand for PATCH-requests
     * @param string $url
     * @param array $payload
     * @return Response
     * @throws \Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function patch(string $url, array $payload = null)
    {
        $request = new PatchRequest($url, $payload);

        return $this->sendRequest($request);
    }

    /**
     * Shorthand for DELETE-requests
     * @param string $url
     * @return Response
     * @throws \Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function delete(string $url)
    {
        $request = new DeleteRequest($url);

        return $this->sendRequest($request);
    }

    /**
     * Adds the necessary headers for the API call to the guzzle options array
     *
     * @param array &$guzzleOptions
     * @param Request $request
     */
    protected function addHeaders(array &$guzzleOptions, Request $request)
    {
        $headers = [
            'X-Authorization' => 'Bearer ' . $this->apiToken
        ];

        if ($request->isPaginatable() && null !== $request->getPage()) {
            $headers = array_merge($headers, $request->getRangeHeaders());
        }

        if ($request->isPayloadable()) {
            $guzzleOptions['json'] = $request->getPayload();
        }

        $guzzleOptions['headers'] = array_merge($guzzleOptions['headers'] ?? [], $headers);
    }

    /**
     * Builds the key used by the cache to store/retrieve request data
     * @param Request $request
     * @param $endpointUrl
     * @return string
     */
    protected function getCacheKey(Request $request, $endpointUrl)
    {
        $cacheKeyParts = [
            $request->getRequestMethod(),
            $endpointUrl
        ];

        if ($request->isPaginatable() && null !== $request->getPage()) {
            $cacheKeyParts[] = $request->getRangeString();
        }

        return 'bamboo_' . hash('sha256', preg_replace('/[^0-9a-z-_\.]/i', "-", implode('-', $cacheKeyParts)));
    }
}