<?php

namespace Tests\Fatpanda\BambooConnector;

use GuzzleHttp\Psr7\Response as GuzzleResponse;
use PHPUnit\Framework\TestCase;
use Fatpanda\BambooConnector\HttpFoundation\Response;

class ResponseTest extends TestCase
{
    public function test()
    {
        $guzzleResponse = new GuzzleResponse('200', ['Content-Range' => '8-11/58', 'Range-Unit' => 'items'], \GuzzleHttp\json_encode(['test' => 'value']));
        $response = new Response($guzzleResponse);

        $this->assertEquals('200', $response->getStatusCode());
        $this->assertEquals('OK', $response->getStatusMessage());
        $this->assertEquals([
            'test' => 'value'
        ], $response->getData());
        $this->assertEquals('8-11', $response->getRange());
        $this->assertEquals('58', $response->getCollectionCount());

        $this->assertEquals('{"test":"value"}', $response->getData(Response::DESERIALIZATION_TYPE_NONE));
    }

    public function test_dataStructure()
    {
        $guzzleResponse = new GuzzleResponse('200', [], \GuzzleHttp\json_encode(['data' => ['test' => 'value']]));
        $response = new Response($guzzleResponse);

        $this->assertEquals('200', $response->getStatusCode());
        $this->assertEquals('OK', $response->getStatusMessage());
        $this->assertEquals([
            'test' => 'value'
        ], $response->getData(Response::DESERIALIZATION_TYPE_ARRAY));
        $this->assertNull($response->getRange());

        $this->assertEquals('{"data":{"test":"value"}}', $response->getData(Response::DESERIALIZATION_TYPE_NONE));
    }

    public function test_serialize()
    {
        $guzzleResponse = new GuzzleResponse('200', ['Content-Range' => '8-11/41', 'Range-Unit' => 'items'], \GuzzleHttp\json_encode(['test' => 'value']));
        $response = new Response($guzzleResponse);

        $serialized = serialize($response);
        $this->assertEquals('C:48:"Fatpanda\BambooConnector\HttpFoundation\Response":75:{a:4:{i:0;i:200;i:1;s:2:"OK";i:2;s:16:"{"test":"value"}";i:3;s:7:"8-11/41";}}', $serialized);
    }

    public function test_unserialize()
    {
        /** @var Response $unserialized */
        $unserialized = unserialize('C:48:"Fatpanda\BambooConnector\HttpFoundation\Response":75:{a:4:{i:0;i:200;i:1;s:2:"OK";i:2;s:16:"{"test":"value"}";i:3;s:7:"8-11/44";}}');

        $this->assertEquals(200, $unserialized->getStatusCode());
        $this->assertEquals('OK', $unserialized->getStatusMessage());
        $this->assertEquals(['test' => 'value'], $unserialized->getData());
        $this->assertEquals('8-11', $unserialized->getRange());
        $this->assertEquals('44', $unserialized->getCollectionCount());
    }

    public function test_invalidDeserializationType()
    {
        $guzzleResponse = new GuzzleResponse('200', ['Content-Range' => '8-11/88', 'Range-Unit' => 'items'], \GuzzleHttp\json_encode(['test' => 'value']));
        $response = new Response($guzzleResponse);

        $this->expectException(\InvalidArgumentException::class);
        $response->getData('INVLAID');
    }
}