<?php

namespace Fatpanda\BambooConnector\HttpFoundation;

use \GuzzleHttp\Psr7\Response as GuzzleResponse;

class Response implements \Serializable
{
    /** @var int  */
    private $statusCode;

    /** @var string  */
    private $statusMessage;

    /** @var string  */
    private $responseBody;

    /** @var null|array */
    private $range;

    const DESERIALIZATION_TYPE_NONE = 'none';
    const DESERIALIZATION_TYPE_ARRAY = 'array';

    const DESERIALIZATION_TYPES = [
        self::DESERIALIZATION_TYPE_NONE,
        self::DESERIALIZATION_TYPE_ARRAY
    ];

    /**
     * Response constructor.
     * @param GuzzleResponse $response
     */
    public function __construct(GuzzleResponse $response)
    {
        $this->statusCode = $response->getStatusCode();
        $this->statusMessage = $response->getReasonPhrase();
        $this->responseBody = (string) $response->getBody();

        $rangeHeader = $response->getHeaderLine('Range');
        if ($rangeHeader !== '') {
            $this->range = $rangeHeader;
        }
    }

    /**
     * Handles deserialization depending on $deserializationType.
     * @param string $deserializationType
     * @return array|null|string
     */
    public function getData(string $deserializationType = self::DESERIALIZATION_TYPE_ARRAY)
    {
        if (!in_array($deserializationType, self::DESERIALIZATION_TYPES)) {
            throw new \InvalidArgumentException(sprintf('Invalid deserialization type: %s', $deserializationType));
        }

        // if the API returned an empty response without throwing an error,
        // an empty value is probably allowed
        if (empty($this->responseBody)) {
            return null;
        }

        switch ($deserializationType) {
            case self::DESERIALIZATION_TYPE_NONE:
                return $this->responseBody;
                break;

            case self::DESERIALIZATION_TYPE_ARRAY:
                return $this->jsonDecode($this->responseBody);
                break;
        }
    }

    /**
     * @return int
     */
    public function getStatusCode()
    {
        return $this->statusCode;
    }

    /**
     * @return string
     */
    public function getStatusMessage()
    {
        return $this->statusMessage;
    }

    /**
     * @return array|null|string
     */
    public function getRange()
    {
        return $this->range;
    }

    /**
     * @return string
     */
    public function serialize()
    {
        return serialize([
            $this->statusCode,
            $this->statusMessage,
            $this->responseBody,
            $this->range
        ]);
    }

    /**
     * @param string $serialized
     */
    public function unserialize($serialized)
    {
        list(
            $this->statusCode,
            $this->statusMessage,
            $this->responseBody,
            $this->range
            ) = unserialize($serialized);
    }

    /**
     * Wraps json_decode to handle the data wrapper property
     * @param string $jsonString
     * @return array
     */
    private function jsonDecode(string $jsonString)
    {
        $data = \GuzzleHttp\json_decode($jsonString, true);

        // normalize data in case of "{'data': [...]}" structure
        if (array_key_exists('data', $data)) {
            return $data['data'];
        }

        return $data;
    }
}